/*
 * cGroup.cpp
 *
 * Implementation of group.
 */
#include "cGroup.h"

#include "Util/cException.h"
#include "Endpoint/cEndpoint.h"


/*
 * cGroup::cGroup()
 *
 * Purpose:	Creates a group that can hold up to "size" members.
 * IN:		size		-> The maximum number of entries.
 * Cond:	-
 * PostCnd:	The group list is set up.
 * Return:	-
 */
cGroup::cGroup(int size)
{
	mList = NULL;
	mTotalSize = 0;
	mList = new cFifo(size);
	if(!mList)
	{
		throw cException("<cGroup::cGroup>: Unable to alloc the list structure.");
	}
}

/*
 * cGroup::~cGroup()
 *
 * Purpose:	Destructor
 */
cGroup::~cGroup()
{
	Lock();

	if(mList)
	{
		Clear();	// Release all endpoints.
		Unlock();
		delete mList;
	}
}

/*
 * cGroup::AddEndpoint()
 *
 * Purpose:	Adds an endpoint to the group list.
 * IN:		endpoint	-> The endpoint to add.
 * Cond:	-
 * PostCnd:	The endpoint is added to the registration list.
 * Return:	true if success, else false.
 */
bool cGroup::AddEndpoint(cEndpoint* endpoint)
{
	Node *node;
	endpoint->AddRef();
	//node = mList->Push((cObject *)endpoint));
	node = mList->Push((cObject *)endpoint);
	if(!node)
	{
		endpoint->Release();
		return false;
	}
	else
	{
		mTotalSize = mTotalSize + endpoint->GetSize();
		return true;
	}
}

/*
 * cGroup::IsMember()
 *
 * Purpose:	Checks to see if the endpoint is a member of the group.
 * IN:		endpoint	-> The endpoint (or one just like it) to check.
 * Cond:	-
 * PostCnd:	-
 * Return:	true if success, otherwise false.
 */
bool	cGroup::IsMember(cEndpoint* endpoint)
{
	Lock();

	cEndpoint* testEp;
	cIterator* iter = GetIterator();
	while(!iter->Done())
	{
		testEp = (cEndpoint *)(iter->GetData());
		if(testEp->Equals(endpoint))
		{
			Unlock();
			return true;
		}
		iter->GetNext();
	}
	Unlock();
	return false;
}

/*
 * cGroup::RemoveEndpoint()
 *
 * Purpose:	Removes the first similar endpoint from the group.
 * IN:		endpoint	-> The endpoint (or one just like it) to remove.
 * Cond:	endpoint should be in the group.
 * PostCnd:	endpoint is removed from the group.
 * Return:	true if success, otherwise false.
 */
bool	cGroup::RemoveEndpoint(cEndpoint* endpoint)
{
	cEndpoint* testEp;
	Lock();

	cIterator* iter = GetIterator();
	while(!iter->Done())
	{
		testEp = (cEndpoint *)iter->GetData();
		if(testEp->Equals(endpoint))
		{
			mTotalSize = mTotalSize - testEp->GetSize();
			testEp->Release();
			Unlock();
			return iter->DeleteCurrent();
		}
		iter->GetNext();
	}
	Unlock();
	return false;
}

/*
 * cGroup::Clear()
 *
 * Purpose:	Removes all endpoints.
 * IN:		-
 * Cond:	-
 * PostCnd:	The group is cleared.
 * Return:	true if success, otherwise false.
 */
bool cGroup::Clear()
{
	cEndpoint *ep;
	Lock();

	cIterator* iter = GetIterator();
	while(!iter->Done())
	{
		ep = (cEndpoint *)iter->GetData();
		ep->Release();
		iter->DeleteCurrent();	// Automatically moves on to next
	}
	mTotalSize = 0;
	Unlock();
	return true;
}

void cGroup::Print(ostream &errStream)
{
	Lock();
	cIterator* iter = GetIterator();
	while(!iter->Done())
	{
		((cEndpoint *)iter->GetData())->Print(errStream);
		iter->GetNext();
	}
	Unlock();
}

unsigned int cGroup::GetNumElements()
{
	return (unsigned)mList->GetNumElements();
}

cIterator* cGroup::GetIterator()
{
	mIterator = mList->GetIterator();
	return (cIterator *)&mIterator; 
}

bool	cGroup::Lock()
{
	mList->Lock();
	return true;
}

bool	cGroup::Unlock()
{
	mList->Unlock();
	return true;
}